# =================================================================================
# OPENHousing User Feedback and Diagnostic Capture Tool (V15 - Full Report Fix)
#
# PURPOSE: A final, robust version that correctly populates the local .txt file
#          with the full diagnostic report content.
#
# Version: 15.1
# =================================================================================

# --- 1. CONFIGURATION ---
$openHousingLogPath = "C:\ProgramData\OPENHousing\Logs\app.log"
$apiUrl = "https://report-oh.westnorthants.digital/submit_report"

# --- 2. INITIALIZATION & ADMIN CHECK ---
try { Add-Type -AssemblyName PresentationFramework, System.Drawing, System.Windows.Forms } catch { Write-Error "Failed to load required .NET Assemblies."; exit }
$isAdmin = ([System.Security.Principal.WindowsPrincipal][System.Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([System.Security.Principal.WindowsBuiltInRole]::Administrator)
$captureMode = if ($isAdmin) { "Administrator" } else { "Standard User" }

# --- 3. DEFINE THE WPF INTERFACE (XAML is identical) ---
[xml]$xaml = @"
<Window xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
        xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
        Title="OPENHousing Feedback &amp; Diagnostics Tool" Height="640" Width="800"
        WindowStartupLocation="CenterScreen" ResizeMode="NoResize" FontFamily="Segoe UI">
    <Grid>
        <Grid.ColumnDefinitions><ColumnDefinition Width="2*" /><ColumnDefinition Width="1*" /></Grid.ColumnDefinitions>
        <StackPanel Grid.Column="0" Margin="20">
            <Label Content="Your Name: *" FontWeight="Bold"/><TextBox x:Name="NameTextBox" Padding="2" Margin="0,0,0,10"/>
            <Label Content="Your Location: *" FontWeight="Bold"/><ComboBox x:Name="LocationComboBox" Padding="2" Margin="0,0,0,10" IsEditable="False"/>
            <Label Content="Details (What were you trying to do? Which modules?): *" FontWeight="Bold"/><TextBox x:Name="DetailsTextBox" Height="120" Padding="2" Margin="0,0,0,10" TextWrapping="Wrap" AcceptsReturn="True" VerticalScrollBarVisibility="Auto"/>
            <Grid Margin="0,0,0,10">
                <Grid.ColumnDefinitions><ColumnDefinition Width="*" /><ColumnDefinition Width="*" /></Grid.ColumnDefinitions>
                <StackPanel Grid.Column="0" Margin="0,0,10,0"><Label Content="Date of Issue:" FontWeight="Bold"/><DatePicker x:Name="DatePicker"/></StackPanel>
                <StackPanel Grid.Column="1" Margin="10,0,0,0">
                    <Label Content="Time of Issue (24-hour format):" FontWeight="Bold"/>
                    <Grid>
                        <Grid.ColumnDefinitions><ColumnDefinition Width="*"/><ColumnDefinition Width="Auto"/><ColumnDefinition Width="*"/></Grid.ColumnDefinitions>
                        <ComboBox x:Name="HourComboBox" Grid.Column="0" Margin="0,0,5,0" VerticalContentAlignment="Center"/>
                        <TextBlock Grid.Column="1" Text=":" VerticalAlignment="Center" FontWeight="Bold"/>
                        <ComboBox x:Name="MinuteComboBox" Grid.Column="2" Margin="5,0,0,0" VerticalContentAlignment="Center"/>
                    </Grid>
                </StackPanel>
            </Grid>
            <Label Content="Error Message Seen (Optional):" FontWeight="Bold"/><TextBox x:Name="ErrorTextBox" Padding="2" Margin="0,0,0,15"/>
            <Button x:Name="SubmitButton" Content="Submit Report" Height="40" FontWeight="Bold" FontSize="14" Background="#007ACC" Foreground="White"/>
            <ProgressBar x:Name="ProgressBar" Height="20" Margin="0,10,0,5" Visibility="Hidden"/>
            <TextBlock x:Name="StatusText" TextAlignment="Center" Opacity="0.7"/>
        </StackPanel>
        <Border Grid.Column="1" Background="#f0f0f0" Padding="15"><StackPanel>
            <Label Content="Diagnostics to be Captured" FontWeight="Bold" FontSize="16" Margin="0,0,0,10"/>
            <TextBlock x:Name="DiagnosticsList" TextWrapping="Wrap" LineHeight="20" />
        </StackPanel></Border>
    </Grid>
</Window>
"@

# --- 4. CREATE, MAP, AND POPULATE CONTROLS (Identical) ---
$reader = New-Object System.Xml.XmlNodeReader $xaml; $window = [System.Windows.Markup.XamlReader]::Load($reader)
$nameTextBox = $window.FindName("NameTextBox"); $locationComboBox = $window.FindName("LocationComboBox"); $detailsTextBox = $window.FindName("DetailsTextBox"); $datePicker = $window.FindName("DatePicker"); $hourComboBox = $window.FindName("HourComboBox"); $minuteComboBox = $window.FindName("MinuteComboBox"); $errorTextBox = $window.FindName("ErrorTextBox"); $submitButton = $window.FindName("SubmitButton"); $progressBar = $window.FindName("ProgressBar"); $statusText = $window.FindName("StatusText"); $diagnosticsList = $window.FindName("DiagnosticsList")
$nameTextBox.Text = $env:USERNAME; $datePicker.Text = (Get-Date).ToShortDateString()
@("Home", "Office - OAS", "Office - Guildhall", "Office - Westbridge", "Office - Other") | ForEach-Object { $locationComboBox.Items.Add($_) }; $locationComboBox.SelectedIndex = 0
$diagnosticsList.Text = "The following information will be automatically collected:`n`n VDI Hostname & Uptime`n User's Physical Computer Name`n Overall CPU & Memory Usage`n OPENHousing Process Details`n Recent Application Log Entries`n Full Process List"
0..23 | ForEach-Object { $hourComboBox.Items.Add($_.ToString("00")) }; 0..59 | ForEach-Object { $minuteComboBox.Items.Add($_.ToString("00")) }
$currentTime = Get-Date; $hourComboBox.SelectedItem = $currentTime.ToString("HH"); $minuteComboBox.SelectedItem = $currentTime.ToString("mm")

# --- 5. LOGIC FOR SUBMIT BUTTON ---
$submitButton.add_Click({
    # Validation
    if ([string]::IsNullOrWhiteSpace($nameTextBox.Text)) { [System.Windows.MessageBox]::Show("The 'Your Name' field cannot be empty.", "Validation Error", "OK", "Warning"); return }
    if ([string]::IsNullOrWhiteSpace($detailsTextBox.Text)) { [System.Windows.MessageBox]::Show("The 'Details' field cannot be empty. Please describe the issue.", "Validation Error", "OK", "Warning"); return }
    
    # Update UI
    $submitButton.IsEnabled = $false; $progressBar.Visibility = "Visible"; $progressBar.Value = 0; $window.Cursor = [System.Windows.Input.Cursors]::Wait
    
    function Update-Progress ($value, $status) { $progressBar.Value = $value; $statusText.Text = $status; $window.Dispatcher.Invoke([action]{}, "Normal") }

    # --- Gather Data ---
    Update-Progress 10 "Gathering basic info..."; $vdiName = $env:COMPUTERNAME; $captureTime = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    Update-Progress 25 "Gathering session info..."; try { $clientName = $env:CLIENTNAME; $uptime = (Get-CimInstance Win32_OperatingSystem).LastBootUpTime; $sessionUptime = New-TimeSpan -Start $uptime; $sessionUptimeString = "$($sessionUptime.Days)d $($sessionUptime.Hours)h $($sessionUptime.Minutes)m"; $sessionReport = "User's Physical Computer Name: $clientName`nVDI Uptime: $sessionUptimeString" } catch { $sessionReport = "User's Physical Computer Name: $env:CLIENTNAME`nVDI Uptime: Could not be determined." }
    Update-Progress 50 "Capturing system resources..."; try { $memory = Get-CimInstance Win32_OperatingSystem | Select-Object TotalVisibleMemorySize, FreePhysicalMemory; $totalMemoryMB = [math]::Round($memory.TotalVisibleMemorySize / 1024); $freeMemoryMB = [math]::Round($memory.FreePhysicalMemory / 1024); $cpu = Get-CimInstance Win32_Processor | Measure-Object -Property LoadPercentage -Average; $cpuUsage = [math]::Round($cpu.Average); $resourceReport = "CPU Usage: $cpuUsage %`nMemory: $freeMemoryMB MB Free of $totalMemoryMB MB Total" } catch { $resourceReport = "Could not retrieve CPU/Memory stats (Access Denied)." }
    Update-Progress 70 "Analyzing OPENHousing process..."; $openHousingProcesses = Get-Process -Name "prowin32" -ErrorAction SilentlyContinue; $appSpecificReport = ""; if ($openHousingProcesses) { foreach ($proc in $openHousingProcesses) { $procUptime = New-TimeSpan -Start $proc.StartTime; $procUptimeString = "$($procUptime.Days)d $($procUptime.Hours)h $($procUptime.Minutes)m"; $appSpecificReport += "  - PID: $($proc.Id)`n    Memory (MB): $([math]::Round($proc.WS / 1MB))`n    Running For: $procUptimeString`n`n" } } else { $appSpecificReport = "No 'prowin32.exe' process was found." }
    Update-Progress 85 "Reading application logs..."; try { if (Test-Path $openHousingLogPath) { $logTail = Get-Content $openHousingLogPath -Tail 50 | Out-String } else { $logTail = "Log file not found at: $openHousingLogPath" } } catch { $logTail = "Could not read log file (Access Denied)." }
    Update-Progress 95 "Getting full process list..."; $processList = Get-Process | Sort-Object -Property WS -Descending | Format-Table -AutoSize | Out-String
    
    # Assemble Report Data
    $safeUserName = $nameTextBox.Text -replace '[\\/:*?"<>|]', ''; $timestamp = Get-Date -Format "yyyyMMdd-HHmmss";
    $selectedDateString = $datePicker.Text; $issueDateTime = "$($selectedDateString) $($hourComboBox.SelectedItem):$($minuteComboBox.SelectedItem)"
    
    # Build the JSON Payload
    $payload = @{
        userName = $nameTextBox.Text
        userLocation = $locationComboBox.SelectedItem
        issueDateTime = $issueDateTime
        issueDetails = $detailsTextBox.Text
        errorMessage = $errorTextBox.Text
        diagnostics = @{
            captureMode = $captureMode
            vdiHostname = $vdiName
            sessionInfo = $sessionReport
            resourceInfo = $resourceReport
            appSpecificInfo = $appSpecificReport
            appLog = $logTail
            fullProcessList = $processList
        }
    }
    $jsonData = $payload | ConvertTo-Json -Depth 5

    # Try to send data to server
    Update-Progress 100 "Submitting report to server..."
    $submissionSuccess = $false
    try {
        Invoke-RestMethod -Uri $apiUrl -Method Post -Body $jsonData -ContentType "application/json" -TimeoutSec 30
        $submissionSuccess = $true
    } catch {
        $submissionErrorMessage = $_.Exception.Message
    }

    # Save the local .txt file REGARDLESS of submission success
    $savePath = $PSScriptRoot
    $fileName = "OPENHousing_Report_$($safeUserName)_$($timestamp).txt"
    $filePath = Join-Path -Path $savePath -ChildPath $fileName
    
    # --- FIX: Build the full text file content ---
    $reportContent = @"
=========================================================
      OPENHousing User Feedback Report
=========================================================

---------------------------------------------------------
           USER-PROVIDED DETAILS
---------------------------------------------------------
Name: $($nameTextBox.Text)
Location: $($locationComboBox.SelectedItem)
Date/Time of Issue: $($issueDateTime)

Details of Work Being Done:
$($detailsTextBox.Text)

Error Message Seen:
$($errorTextBox.Text)

=========================================================
          AUTOMATICALLY CAPTURED DIAGNOSTICS
=========================================================
Capture Mode: $($captureMode)
---------------------------------------------------------
VDI SESSION & RESOURCE INFO
---------------------------------------------------------
Report Capture Time: $($captureTime)
VDI Hostname: $($vdiName)
$($sessionReport)
$($resourceReport)

---------------------------------------------------------
APPLICATION-SPECIFIC DETAILS (prowin32.exe)
---------------------------------------------------------
$($appSpecificReport)

---------------------------------------------------------
RECENT APPLICATION LOG ENTRIES
---------------------------------------------------------
$($logTail)

---------------------------------------------------------
FULL TASK MANAGER PROCESS LIST
---------------------------------------------------------
(Sorted by Memory Usage at time of report)

$($processList)
"@
    
    Set-Content -Path $filePath -Value $reportContent

    # Provide a dynamic success/fail message to the user
    $window.Cursor = [System.Windows.Input.Cursors]::Arrow
    if ($submissionSuccess) {
        $message = "Thank you! Your report was sent successfully.`n`nA backup copy has also been saved to the script folder as:`n`n$fileName"
        [System.Windows.MessageBox]::Show($message, "Submission Successful", "OK", "Information")
    } else {
        $message = "Could not send the report to the server. The server may be offline.`n`nIMPORTANT: A backup copy of your report has been saved to the script folder as:`n`n$fileName`n`nPlease email this file to the IT support team.`n`nError: $submissionErrorMessage"
        [System.Windows.MessageBox]::Show($message, "Submission Failed", "OK", "Warning")
    }
    $window.Close()
})

# --- 6. SHOW THE WINDOW ---
$window.ShowDialog() | Out-Null